package macros.builders.s1

import macros.builders.s2.Person

case class Person private[s1] (name: String, age: Int, email: String)

case class PersonBuilder[
  N <: Option[String],
  A <: Option[Int],
  E <: Option[String]
] private (
  private[s1] val nameOpt: N,
  private[s1] val ageOpt: A,
  private[s1] val emailOpt: E
) {
  private[s1] def setName(name: String): PersonBuilder[Some[String], A, E] =
    copy[Some[String], A, E](nameOpt = Some(name))

  private[s1] def setAge(age: Int): PersonBuilder[N, Some[Int], E] =
    copy(ageOpt = Some(age))

  private[s1] def setEmail(email: String): PersonBuilder[N, A, Some[String]] =
    copy(emailOpt = Some(email))
}

object PersonBuilder {
  def apply(): PersonBuilder[None.type, None.type, None.type] =
    new PersonBuilder(None, None, None)

  implicit final class NameOps[A <: Option[Int], E <: Option[String]](
    private val builder: PersonBuilder[None.type, A, E]
  ) extends AnyVal {
    def name(value: String): PersonBuilder[Some[String], A, E] =
      builder.setName(value)
  }

  implicit final class AgeOps[N <: Option[String], E <: Option[String]](
    private val builder: PersonBuilder[N, None.type, E]
  ) extends AnyVal {
    def age(value: Int): PersonBuilder[N, Some[Int], E] =
      builder.setAge(value)
  }

  implicit final class EmailOps[N <: Option[String], A <: Option[Int]](
    private val builder: PersonBuilder[N, A, None.type]
  ) extends AnyVal {
    def email(value: String): PersonBuilder[N, A, Some[String]] =
      builder.setEmail(value)
  }

  implicit final class BuildDoneOps(
    private val builder: PersonBuilder[Some[String], Some[Int], Some[String]]
  ) extends AnyVal {
    def build(): Person =
      Person(
        builder.nameOpt.value,
        builder.ageOpt.value,
        builder.emailOpt.value
      )
  }

  def main(args: Array[String]): Unit = {
    println(
      PersonBuilder()
        .name("John")
        .age(30)
        .email("hello@world.com")
        .build()
    )
  }
}
